import java.io.*;
import java.util.*;

/**
 * Generates Java Rupy Services from HTML pages with [[ ]] as mark-up.
 * TODO: multi-line comments and maybe level more than once per line?
 * @author marc.larue
 */
public class Page {
	static final String START = "[[";
	static final String STOP = "]]";
	static final String SUFFIX = ".html";

	static final int HTML = 1;
	static final int JAVA = 2;

	static long time;
	static int state, level;
	static Filter filter;

	public static void main(String[] args) {
		if (args.length > 1) {
			try {
				time = new File(args[0]).lastModified();
				File path = new File(args[1]);
				filter = new Filter();
				
				if(path.isDirectory()) {
					Iterator it = files(path).iterator();

					while(it.hasNext()) {
						generate((File) it.next());
					}
				}
				else {
					System.out.println("Path '" + args[0] + "' needs to be a directory.");
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		} else {
			System.out.println("Usage: Page [time] [path]\nWhere [time] is a file against which you compare to see which files have changed since the last compile and [path] is the path to the folder containing your .html pages.");
		}
	}
	
	static void generate(File html) throws IOException {
		String path = path(html) + File.separator + "gen";
		File java = new File(path + File.separator + name(html) + ".java");

		new File(path).mkdirs();

		System.out.println(java);
		
		BufferedReader in = new BufferedReader(new InputStreamReader(new FileInputStream(html), "UTF-8"));
		PrintStream out = new PrintStream(new FileOutputStream(java, false), true, "UTF-8");

		boolean found = false;
		String line = null;

		state = HTML;
		level = 0;

		while((line = in.readLine()) != null) {
			if(parse(out, line)) {
				found = true;
			}
		}

		in.close();
		out.flush();
		out.close();

		if(!found) {
			java.delete();
		}
	}

	static boolean parse(PrintStream out, String line) {
		int start = line.indexOf(START);
		int stop = line.indexOf(STOP);

		//System.out.println(line + " " + level);

		if(start > -1 && stop > -1) {
			while(start > -1 || stop > -1) {
				/* Should maybe inline all Java that is wrapped by HTML
				 * so that developers can keep line info if they wish.
				 * But what happens if someone writes non-inlineable code?
				 */
				if(start > -1 && stop > -1) {
					if(start > stop) {
						line = stop(out, line, stop);
					}
					else {
						line = start(out, line, start);
					}
				}
				else if(start > -1) {
					start(out, line, start);
					return true;
				}
				else if(stop > -1) {
					stop(out, line, stop);
					return true;
				}
				else {
					println(out, line, state);
					return true;
				}

				start = line.indexOf(START);
				stop = line.indexOf(STOP);
			}
		}
		else if(start > -1) {
			start(out, line, start);
		}
		else if(stop > -1) {
			stop(out, line, stop);
		}
		else {
			println(out, line, state);
			return false;
		}

		return true;
	}

	static String start(PrintStream out, String line, int start) {
		int index = line.indexOf(STOP, start);

		String html = line.substring(0, start);
		String java = line.substring(start + START.length(), index > -1 ? index : line.length());

		print(out, html, state);

		state = JAVA;

		if(index > -1 && line.length() > index + STOP.length()) {
			print(out, java, state);
		}
		else {
			println(out, java, state);
		}

		return index > -1 ? line.substring(index) : null;
	}

	static String stop(PrintStream out, String line, int stop) {
		int index = line.indexOf(START, stop);

		String java = line.substring(0, stop);
		String html = line.substring(stop + STOP.length(), index > -1 ? index : line.length());

		print(out, java, state);

		state = HTML;

		if(index > -1 && line.length() > index + START.length()) {
			print(out, html, state);
		}
		else {
			println(out, html, state);
		}

		return index > -1 ? line.substring(index) : null;
	}

	static void print(PrintStream out, String line, int type) {
		print(out, line, type, false);
	}

	static void println(PrintStream out, String line, int type) {
		print(out, line, type, true);
	}

	static void print(PrintStream out, String line, int type, boolean ln) {
		if(line.length() > 0) {
			switch(type) {
			case HTML:
				if(line.trim().length() > 0) {
					out.println(pad(level) + "out.print" + (ln ? "ln" : "") + "(\"" + replace(line) + "\");");
				}
				break;
			case JAVA:
				line = line.trim();
				if(line.length() > 0) {
					int code = code(line);

					//System.out.println(code + " " + line);

					if(code > 0) {
						if(code == 100) { level--; }
						if(code == 300) {
							out.println(pad(level - 1) + line);
						}
						else {
							out.println(pad(level) + line);
						}
						if(code == 200) { level++; }
					}
					else {
						out.println(pad(level) + "out.print" + (ln ? "ln" : "") + "(" + line + ");");
					}
				}
				break;
			}
		}
	}

	static int code(String line) {
		int open = count(line, '{');
		int close = count(line, '}');

		if(open - close < 0) { return 100; }
		if(open - close > 0) { return 200; }

		if(open > 0 || close > 0) {
			if(line.indexOf('{') > line.indexOf('}')) {
				return 300;
			}
			else {
				return 400;
			}
		}

		String trim = line.trim();
		
		if(trim.startsWith("//") || trim.indexOf("/*") > -1 || trim.indexOf("*/") > -1) {
			return 500;
		}

		if(line.indexOf(";") > -1) {
			return 600;
		}

		if(line.indexOf("if") > -1) {
			return 700;
		}

		if(line.trim().endsWith("+")) {
			return 800;
		}

		if(line.trim().endsWith("&&")) {
			return 900;
		}

		if(line.trim().endsWith("||")) {
			return 1000;
		}
		
		//if(line.indexOf(".") > -1) {
		//	return 1100;
		//}
		
		return 0;
	}

	static int count(String line, char character) {
		int count = 0;
		int index = line.indexOf(character, 0);

		while(index > -1) {
			count++;
			index = line.indexOf(character, index + 1);
		}

		return count;
	}
	/*
	 * \" -> \\\"
	 * \n -> \\n
	 * etc.
	 */
	static String replace(String line) {
		int index = line.indexOf("\\");

		while(index > -1) {
			line = line.substring(0, index) + "\\" + line.substring(index, line.length());
			index = line.indexOf("\\", index + 2);
		}
		
		line = replace(line, "\"", "\\\"");
		
		return line;
	}
	static String replace(String line, String from, String to) {
		int index = line.indexOf(from);

		while(index > -1) {
			line = line.substring(0, index) + to + line.substring(index + from.length());
			index = line.indexOf(from, index + to.length());
		}

		return line;
	}

	static String pad(int level) {
		StringBuffer padding = new StringBuffer();

		for(int i = 0; i < level; i++) {
			padding.append("    ");
		}

		return padding.toString();
	}

	static List files(File dir) throws FileNotFoundException {
		List result = new ArrayList();
		List list = Arrays.asList(dir.listFiles(filter));

		Iterator it = list.iterator();

		while(it.hasNext()) {
			File file = (File) it.next();

			if(file.isFile()) {
				result.add(file);
			}
			else if(file.isDirectory()) {
				result.addAll(files(file));
			}
		}

		return result;
	}

	static String path(File file) {
		String name = file.getPath();
		name = name.substring(0, name.lastIndexOf(File.separator));
		return name;
	}

	static String name(File file) {
		String name = file.getPath();
		name = name.substring(name.lastIndexOf(File.separator) + 1, name.indexOf("."));
		return name;
	}

	static class Filter implements FilenameFilter {
		public boolean accept(File dir, String name) {
			File file = new File(dir + File.separator + name);
			
			if (name.endsWith(SUFFIX)) {
				if(file.lastModified() > time) {
					return true;
				}
				else {
					File gen = new File(dir + File.separator + "gen");
					if(!gen.exists() || !gen.isDirectory()) {
						return true;
					}
				}
			}
			
			if (file.isDirectory()) {
				return true;
			}

			return false;
		}
	}
}
